<div class="modal" id="historyModal">
  <div class="modal-overlay"></div>
  <div class="modal-container bet-history-modal-container">
    <div class="modal-header bet-history-header">
      <h2>MY BET HISTORY</h2>
      <button class="modal-close" onclick="closeModal('historyModal')">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <line x1="18" y1="6" x2="6" y2="18"></line>
          <line x1="6" y1="6" x2="18" y2="18"></line>
        </svg>
      </button>
    </div>
    <div class="modal-body bet-history-body">
      <!-- Column Headers -->
      <div class="bet-history-columns">
        <div class="history-col-header">Date</div>
        <div class="history-col-header">Bet R$ X</div>
        <div class="history-col-header">Cash out R$</div>
        <div class="history-col-header actions-col"></div>
      </div>
      
      <!-- History List -->
      <div class="bet-history-list" id="betHistoryList">
        <div class="loading-text">Loading history...</div>
      </div>
      
      <!-- Load More Button -->
      <div class="bet-history-footer">
        <button class="load-more-btn" id="loadMoreBtn" onclick="loadMoreHistory()" style="display: none;">
          Load more
        </button>
      </div>
    </div>
  </div>
</div>

<script>
  let currentHistoryPage = 1;
  let hasMoreHistory = false;
  const historyPerPage = 10;

  function formatDate(dateString) {
    const date = new Date(dateString);
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = String(date.getFullYear()).slice(-2);
    return `${hours}:${minutes} ${day}-${month}-${year}`;
  }

  function getMultiplierColor(multiplier) {
    const mult = parseFloat(multiplier);
    // Check if multiplier is exactly 2.00 (with small tolerance for floating point)
    if (Math.abs(mult - 2.00) < 0.01) {
      return 'multiplier-purple';
    }
    return 'multiplier-blue';
  }

  function loadHistory(filter = 'all', page = 1) {
    const listContainer = document.getElementById('betHistoryList');
    if (page === 1) {
      listContainer.innerHTML = '<div class="loading-text">Loading history...</div>';
      currentHistoryPage = 1;
    }
    
    fetch(`/api/history?filter=${filter}`)
      .then(async response => {
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
          return response.json();
        } else {
          const text = await response.text();
          console.error('Non-JSON response:', text);
          throw new Error('Server returned an error');
        }
      })
      .then(data => {
        if (data.success && data.history && data.history.length > 0) {
          // Slice for pagination
          const startIndex = (page - 1) * historyPerPage;
          const endIndex = startIndex + historyPerPage;
          const pageData = data.history.slice(startIndex, endIndex);
          hasMoreHistory = endIndex < data.history.length;

          if (page === 1) {
            listContainer.innerHTML = '';
          }

          pageData.forEach(item => {
            const isWin = item.type === 'win' && parseFloat(item.win_amount || 0) > 0;
            const multiplier = parseFloat(item.multiplier || 0).toFixed(2);
            const betAmount = parseFloat(item.bet_amount || 0).toFixed(2);
            const cashOutAmount = parseFloat(item.win_amount || 0).toFixed(2);
            const dateStr = formatDate(item.created_at);
            
            const historyItem = document.createElement('div');
            historyItem.className = `bet-history-item ${isWin ? 'bet-history-win' : ''}`;
            historyItem.innerHTML = `
              <div class="history-col date-col">${dateStr}</div>
              <div class="history-col bet-col">
                <span class="bet-amount">${betAmount}</span>
                <span class="multiplier-badge ${getMultiplierColor(multiplier)}">${multiplier}x</span>
              </div>
              <div class="history-col cashout-col ${isWin ? '' : 'empty-cashout'}">
                ${isWin ? cashOutAmount : '—'}
              </div>
              <div class="history-col actions-col">
                <div class="history-actions">
                  <button class="history-action-btn" title="Provably Fair">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                      <polyline points="9 12 11 14 15 10"/>
                    </svg>
                  </button>
                  <button class="history-action-btn" title="Share">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
                    </svg>
                  </button>
                </div>
              </div>
            `;
            listContainer.appendChild(historyItem);
          });

          // Show/hide load more button
          const loadMoreBtn = document.getElementById('loadMoreBtn');
          if (hasMoreHistory) {
            loadMoreBtn.style.display = 'block';
          } else {
            loadMoreBtn.style.display = 'none';
          }

          // If no items and page 1, show no data message
          if (page === 1 && pageData.length === 0) {
            listContainer.innerHTML = '<div class="no-data">No history found</div>';
          }
        } else {
          if (page === 1) {
            listContainer.innerHTML = '<div class="no-data">No history found</div>';
          }
          document.getElementById('loadMoreBtn').style.display = 'none';
        }
      })
      .catch(error => {
        console.error('Error loading history:', error);
        if (page === 1) {
          listContainer.innerHTML = '<div class="error-text">Error loading history. Please try again.</div>';
        }
      });
  }

  function loadMoreHistory() {
    currentHistoryPage++;
    loadHistory('all', currentHistoryPage);
  }

  // Load history when modal opens
  document.addEventListener('DOMContentLoaded', function() {
    const historyModal = document.getElementById('historyModal');
    if (historyModal) {
      // Use MutationObserver to watch for modal activation
      const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
          if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
            if (historyModal.classList.contains('active')) {
              if (!historyModal.dataset.loaded) {
                loadHistory();
                historyModal.dataset.loaded = 'true';
              }
            } else {
              // Reset when modal closes
              historyModal.dataset.loaded = '';
              currentHistoryPage = 1;
            }
          }
        });
      });
      
      observer.observe(historyModal, {
        attributes: true,
        attributeFilter: ['class']
      });
    }
  });

  // Override openModal to load history if not already loaded
  document.addEventListener('DOMContentLoaded', function() {
    if (window.openModal) {
      const originalOpenModal = window.openModal;
      window.openModal = function(modalId) {
        originalOpenModal(modalId);
        if (modalId === 'historyModal') {
          setTimeout(() => {
            const modal = document.getElementById('historyModal');
            if (modal && !modal.dataset.loaded) {
              loadHistory();
              modal.dataset.loaded = 'true';
            }
          }, 100);
        }
      };
    }
  });
</script>
