/**
 * Chat Module
 * Handles chat functionality including sending messages and random message generation
 */

export function initChat() {
  const chatInput = document.getElementById('chatInput');
  const sendChatBtn = document.getElementById('sendChatBtn');
  const chatMessages = document.getElementById('chat-messages');
  
  if (!chatInput || !sendChatBtn || !chatMessages) return;

  // Random message templates
  const randomMessages = {
    wins: [
      "Finally got a win! 2.5x 💰",
      "Cash out at 1.8x, better than nothing",
      "Just won big! 5.2x 🔥🔥🔥",
      "Small win 1.5x, building up slowly",
      "Cashed out at 3.1x! Yes! 🎉",
      "Need just one big win to recover",
      "2.1x, slowly climbing back",
      "Won 1.9x, patience is key",
      "4.5x baby! 🚀",
      "3.8x, good run today",
      "Cashed at 2.2x, safe play",
      "1.6x but consistent wins",
      "Just recovered my losses! 6.2x 💪",
      "5.8x! On fire today! 🌟",
      "2.4x, steady progress",
      "Huge win! 10.5x 🎰💎",
      "3.5x, doing okay",
      "Cashed at 4.1x, happy with that",
      "1.3x, small but consistent",
      "Won 7.2x! Best today! 🏆",
      "2.8x, recovering nicely",
      "1.9x, take it or leave it",
      "Just doubled up! 2.0x exactly",
      "4.9x! Getting there!",
      "Cashed at 3.3x, good call",
      "5.5x! Finally!",
      "2.7x, patient play pays",
      "3.9x! Nice one",
      "6.8x! 🚀🚀🚀",
      "1.7x, small wins add up",
      "Won 8.2x! 🎉🎉🎉"
    ],
    losses: [
      "Flew away too early 😭",
      "Should have cashed out at 1.5x",
      "Greedy again 😢",
      "Need to control myself",
      "Lost again, trying to recover",
      "This game is rigged sometimes",
      "1.2x and it flew away",
      "Almost made it but no",
      "Lost big today 😭😭",
      "Just want to break even",
      "Always miss the good ones",
      "My luck is terrible today",
      "Another loss, keep trying",
      "1.1x and gone, typical",
      "Why do I keep playing?",
      "Need to stop being greedy",
      "Lost everything back",
      "Starting from zero again",
      "This is frustrating",
      "Can't catch a break",
      "1.3x was my limit",
      "Learning the hard way",
      "Today is not my day",
      "Just one good round please",
      "The pain is real 😭",
      "Keep calm and bet responsibly",
      "Need better discipline"
    ],
    gameplay: [
      "Any predictions for next round?",
      "What's the strategy today?",
      "Think it will go high?",
      "Watching the patterns",
      "Patience is key in this game",
      "Don't chase losses",
      "Set limits and stick to them",
      "Slow and steady wins",
      "Looking for 2x+ multipliers",
      "This game is about timing",
      "Need to focus",
      "Just testing the waters",
      "Small bets for now",
      "Building my bankroll slowly",
      "Anyone cashing out big today?",
      "Let's see what happens",
      "Taking a break after this",
      "New round, new hope",
      "Following the trend",
      "Playing it safe today",
      "What's everyone's strategy?",
      "Need to be more disciplined",
      "Learning as I go",
      "Taking it one round at a time",
      "Small wins are still wins",
      "Don't get greedy",
      "Stay calm everyone",
      "Just here for fun",
      "Trying my luck",
      "Hope for the best"
    ],
    reactions: [
      "Wow! 🎉",
      "Amazing! 🔥",
      "Nice! 💪",
      "Unbelievable! 🚀",
      "Let's go! 🙌",
      "Yes! 🎊",
      "Incredible! 💎",
      "Too good! ⭐",
      "Perfect! ✨",
      "Fantastic! 🌟",
      "Awesome! 🎯",
      "Brilliant! 💫",
      "Great! 👏",
      "Beautiful! 🌈",
      "Stunning! 🏆",
      "Wow wow wow! 🎉🎉",
      "Can't believe it! 😱",
      "So good! 😍",
      "Amazing run! 💰💰",
      "Best ever! 🏅"
    ],
    greetings: [
      "Hey everyone! 👋",
      "Good luck all! 🍀",
      "Let's win together! 💪",
      "New here, wish me luck",
      "Good vibes only! ✨",
      "Hello chat! 👋",
      "Hope today is better",
      "Ready to play! 🎮",
      "Let's make it happen! 🚀",
      "Good evening/morning everyone!",
      "Best of luck! 🍀🍀",
      "Stay positive! 🌟",
      "We can do this! 💪",
      "Keep going everyone!",
      "Stay focused! 🎯"
    ]
  };

  // Random usernames
  const usernames = [
    "Player1", "Winner23", "Lucky7", "GameMaster", "AviatorPro",
    "CashOut", "BetBig", "SkyHigh", "FlyAway", "Multiplier",
    "BetKing", "WinStreak", "ProGamer", "LuckyDraw", "AcePlayer",
    "StarWinner", "GoldenEgg", "RocketMan", "CloudHigh", "BigWin",
    "SwiftCash", "QuickWin", "ProBet", "MasterMind", "LuckyStar",
    "WinnerZone", "BetChamp", "SkyRise", "AviatorX", "MultiplierPro"
  ];

  // Helper function to get random item from array
  function getRandomItem(arr) {
    return arr[Math.floor(Math.random() * arr.length)];
  }

  // Helper function to get random number
  function getRandomNumber(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
  }

  // Helper function to mask username
  function maskUsername(username) {
    if (username.length <= 2) return username;
    const firstChar = username.charAt(0);
    const lastChar = username.charAt(username.length - 1);
    const masked = '*'.repeat(username.length - 2);
    return `${firstChar}${masked}${lastChar}`;
  }

  // Helper function to generate random username
  function generateRandomUsername() {
    const baseName = getRandomItem(usernames);
    const randomNum = getRandomNumber(100, 999);
    return `${baseName}${randomNum}`;
  }

  // Create chat message element
  function createChatMessage(username, text, likeCount = 0) {
    const messageDiv = document.createElement('div');
    messageDiv.className = 'chat-message';
    
    const randomSeed = Math.random().toString(36).substring(7);
    const maskedName = maskUsername(username);
    
    messageDiv.innerHTML = `
      <div class="avatar" style="background-image: url('https://i.pravatar.cc/20?u=${randomSeed}')"></div>
      <div class="chat-content">
        <div class="chat-username">${maskedName}</div>
        ${text ? `<div class="chat-text">${escapeHtml(text)}</div>` : ''}
      </div>
      <div class="chat-reactions">
        <span class="chat-like-count">${likeCount}</span>
        <button class="chat-like-btn">👍</button>
      </div>
    `;
    
    return messageDiv;
  }

  // Escape HTML to prevent XSS
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // Add message to chat
  function addMessage(username, text, likeCount = 0) {
    const message = createChatMessage(username, text, likeCount);
    chatMessages.appendChild(message);
    
    // Scroll to bottom
    chatMessages.scrollTop = chatMessages.scrollHeight;
    
    // Remove old messages if too many (keep last 50)
    while (chatMessages.children.length > 50) {
      chatMessages.removeChild(chatMessages.firstChild);
    }
  }

  // Send user message
  function sendUserMessage() {
    const text = chatInput.value.trim();
    if (!text) return;
    
    // Use "You" as the username for now
    const userName = "You";
    addMessage(userName, text, 0);
    
    // Clear input
    chatInput.value = '';
    
    // Focus back on input for rapid messaging
    chatInput.focus();
  }

  // Generate random message
  function generateRandomMessage() {
    // Choose random category
    const categories = ['wins', 'losses', 'gameplay', 'reactions', 'greetings'];
    const category = getRandomItem(categories);
    
    // Get random message from category
    const messages = randomMessages[category];
    const message = getRandomItem(messages);
    
    // Generate random username
    const username = generateRandomUsername();
    
    // Generate random like count
    const likeCount = getRandomNumber(0, 15);
    
    addMessage(username, message, likeCount);
  }

  // Add initial messages if chat is empty
  function addInitialMessages() {
    // Clear existing messages and add fresh ones
    chatMessages.innerHTML = '';
    
    // Add 15-20 initial random messages
    const initialCount = getRandomNumber(15, 20);
    for (let i = 0; i < initialCount; i++) {
      generateRandomMessage();
    }
  }

  // Auto-generate random messages periodically
  function startAutoMessages() {
    // Generate new message every 1-2 seconds for CRAZY chat activity
    const interval = getRandomNumber(1000, 2000);
    
    setInterval(() => {
      // 95% chance to generate message (very active chat)
      if (Math.random() < 0.95) {
        generateRandomMessage();
      }
    }, interval);
    
    // Also add random bursts of messages
    setInterval(() => {
      if (Math.random() < 0.3) {
        // Generate 2-4 messages at once during burst
        const burstCount = getRandomNumber(2, 4);
        for (let i = 0; i < burstCount; i++) {
          setTimeout(() => {
            generateRandomMessage();
          }, i * 200);
        }
      }
    }, 5000);
  }

  // Event Listeners
  sendChatBtn.addEventListener('click', sendUserMessage);
  
  chatInput.addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
      sendUserMessage();
    }
  });

  // Like button functionality
  chatMessages.addEventListener('click', function(e) {
    if (e.target.classList.contains('chat-like-btn')) {
      const likeCountEl = e.target.parentElement.querySelector('.chat-like-count');
      const currentCount = parseInt(likeCountEl.textContent);
      likeCountEl.textContent = currentCount + 1;
      
      // Add animation
      e.target.style.transform = 'scale(1.2)';
      setTimeout(() => {
        e.target.style.transform = 'scale(1)';
      }, 200);
    }
  });

  // Initialize
  addInitialMessages();
  
  // Start auto messages after 5 seconds
  setTimeout(() => {
    startAutoMessages();
  }, 5000);

  return {
    addMessage,
    generateRandomMessage
  };
}
