<?php
/**
 * Laravel Deployment Diagnostic Script
 * 
 * Upload this file to your project root and access it via browser:
 * https://yourdomain.com/diagnose.php
 * 
 * This will help identify common deployment issues.
 */

// Security: Remove this file after diagnosis!
$allowedIPs = ['127.0.0.1', '::1']; // Add your IP if needed
$clientIP = $_SERVER['REMOTE_ADDR'] ?? '';

// Uncomment to restrict access:
// if (!in_array($clientIP, $allowedIPs)) {
//     die('Access denied');
// }

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Laravel Deployment Diagnostic</title>
    <style>
        body { font-family: monospace; padding: 20px; background: #1a1a1a; color: #0f0; }
        .success { color: #0f0; }
        .error { color: #f00; }
        .warning { color: #ff0; }
        .info { color: #0ff; }
        h1 { color: #fff; }
        h2 { color: #0ff; border-bottom: 1px solid #333; padding-bottom: 5px; }
        pre { background: #000; padding: 10px; border: 1px solid #333; overflow-x: auto; }
        .section { margin: 20px 0; }
    </style>
</head>
<body>
    <h1>🔍 Laravel Deployment Diagnostic</h1>
    <p class="warning">⚠️ <strong>IMPORTANT:</strong> Delete this file after diagnosis for security!</p>
    
    <?php
    $issues = [];
    $warnings = [];
    
    // Check PHP Version
    echo '<div class="section">';
    echo '<h2>1. PHP Version</h2>';
    $phpVersion = PHP_VERSION;
    if (version_compare($phpVersion, '8.1.0', '>=')) {
        echo '<p class="success">✓ PHP Version: ' . $phpVersion . ' (OK)</p>';
    } else {
        echo '<p class="error">✗ PHP Version: ' . $phpVersion . ' (Requires 8.1+)</p>';
        $issues[] = 'PHP version too old';
    }
    echo '</div>';
    
    // Check Required Extensions
    echo '<div class="section">';
    echo '<h2>2. PHP Extensions</h2>';
    $required = ['pdo', 'pdo_mysql', 'mbstring', 'openssl', 'tokenizer', 'xml', 'ctype', 'json', 'bcmath'];
    $missing = [];
    foreach ($required as $ext) {
        if (extension_loaded($ext)) {
            echo '<p class="success">✓ ' . $ext . '</p>';
        } else {
            echo '<p class="error">✗ ' . $ext . ' (MISSING)</p>';
            $missing[] = $ext;
        }
    }
    if (!empty($missing)) {
        $issues[] = 'Missing PHP extensions: ' . implode(', ', $missing);
    }
    echo '</div>';
    
    // Check .env File
    echo '<div class="section">';
    echo '<h2>3. Environment File</h2>';
    if (file_exists('.env')) {
        echo '<p class="success">✓ .env file exists</p>';
        
        $env = parse_ini_file('.env');
        
        // Check APP_KEY
        if (isset($env['APP_KEY']) && !empty($env['APP_KEY']) && strpos($env['APP_KEY'], 'base64:') === 0) {
            echo '<p class="success">✓ APP_KEY is set</p>';
        } else {
            echo '<p class="error">✗ APP_KEY is missing or invalid</p>';
            $issues[] = 'APP_KEY not set - run: php artisan key:generate';
        }
        
        // Check Database Config
        echo '<h3>Database Configuration:</h3>';
        $dbKeys = ['DB_CONNECTION', 'DB_HOST', 'DB_DATABASE', 'DB_USERNAME', 'DB_PASSWORD'];
        foreach ($dbKeys as $key) {
            if (isset($env[$key]) && !empty($env[$key])) {
                $value = $key === 'DB_PASSWORD' ? '***' : $env[$key];
                echo '<p class="success">✓ ' . $key . ' = ' . $value . '</p>';
            } else {
                echo '<p class="error">✗ ' . $key . ' is not set</p>';
                $issues[] = $key . ' not configured';
            }
        }
        
        // Check APP_URL
        if (isset($env['APP_URL']) && !empty($env['APP_URL'])) {
            echo '<p class="success">✓ APP_URL = ' . $env['APP_URL'] . '</p>';
        } else {
            echo '<p class="warning">⚠ APP_URL is not set</p>';
            $warnings[] = 'APP_URL not set';
        }
        
    } else {
        echo '<p class="error">✗ .env file NOT FOUND</p>';
        $issues[] = '.env file missing';
    }
    echo '</div>';
    
    // Check File Permissions
    echo '<div class="section">';
    echo '<h2>4. File Permissions</h2>';
    $dirs = ['storage', 'bootstrap/cache'];
    foreach ($dirs as $dir) {
        if (is_dir($dir)) {
            $perms = substr(sprintf('%o', fileperms($dir)), -4);
            if (is_writable($dir)) {
                echo '<p class="success">✓ ' . $dir . ' is writable (perms: ' . $perms . ')</p>';
            } else {
                echo '<p class="error">✗ ' . $dir . ' is NOT writable (perms: ' . $perms . ')</p>';
                $issues[] = $dir . ' not writable - run: chmod -R 775 ' . $dir;
            }
        } else {
            echo '<p class="error">✗ ' . $dir . ' directory NOT FOUND</p>';
            $issues[] = $dir . ' directory missing';
        }
    }
    echo '</div>';
    
    // Check Storage Link
    echo '<div class="section">';
    echo '<h2>5. Storage Link</h2>';
    if (is_link('public/storage') || file_exists('public/storage')) {
        echo '<p class="success">✓ Storage link exists</p>';
    } else {
        echo '<p class="warning">⚠ Storage link missing</p>';
        $warnings[] = 'Run: php artisan storage:link';
    }
    echo '</div>';
    
    // Check Vendor Directory
    echo '<div class="section">';
    echo '<h2>6. Dependencies</h2>';
    if (is_dir('vendor') && file_exists('vendor/autoload.php')) {
        echo '<p class="success">✓ Composer dependencies installed</p>';
    } else {
        echo '<p class="error">✗ Composer dependencies NOT installed</p>';
        $issues[] = 'Run: composer install';
    }
    echo '</div>';
    
    // Test Database Connection
    echo '<div class="section">';
    echo '<h2>7. Database Connection Test</h2>';
    if (file_exists('.env') && isset($env)) {
        try {
            $host = $env['DB_HOST'] ?? '127.0.0.1';
            $port = $env['DB_PORT'] ?? '3306';
            $database = $env['DB_DATABASE'] ?? '';
            $username = $env['DB_USERNAME'] ?? '';
            $password = $env['DB_PASSWORD'] ?? '';
            
            if (!empty($database) && !empty($username)) {
                $dsn = "mysql:host=$host;port=$port;dbname=$database;charset=utf8mb4";
                $pdo = new PDO($dsn, $username, $password, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                ]);
                
                echo '<p class="success">✓ Database connection successful</p>';
                
                // Test query
                $stmt = $pdo->query("SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = '$database'");
                $result = $stmt->fetch();
                echo '<p class="info">ℹ Found ' . $result['count'] . ' tables in database</p>';
                
            } else {
                echo '<p class="error">✗ Database credentials incomplete</p>';
                $issues[] = 'Database credentials not fully configured';
            }
        } catch (PDOException $e) {
            echo '<p class="error">✗ Database connection FAILED</p>';
            echo '<pre>' . htmlspecialchars($e->getMessage()) . '</pre>';
            $issues[] = 'Database connection error: ' . $e->getMessage();
        }
    } else {
        echo '<p class="warning">⚠ Cannot test - .env file missing</p>';
    }
    echo '</div>';
    
    // Check Laravel Bootstrap
    echo '<div class="section">';
    echo '<h2>8. Laravel Bootstrap</h2>';
    if (file_exists('bootstrap/app.php')) {
        echo '<p class="success">✓ Laravel bootstrap file exists</p>';
    } else {
        echo '<p class="error">✗ Laravel bootstrap file missing</p>';
        $issues[] = 'Laravel installation incomplete';
    }
    echo '</div>';
    
    // Summary
    echo '<div class="section">';
    echo '<h2>📋 Summary</h2>';
    
    if (empty($issues) && empty($warnings)) {
        echo '<p class="success" style="font-size: 18px;">✅ All checks passed! Your configuration looks good.</p>';
    } else {
        if (!empty($issues)) {
            echo '<h3 class="error">Critical Issues:</h3>';
            echo '<ul>';
            foreach ($issues as $issue) {
                echo '<li class="error">' . htmlspecialchars($issue) . '</li>';
            }
            echo '</ul>';
        }
        
        if (!empty($warnings)) {
            echo '<h3 class="warning">Warnings:</h3>';
            echo '<ul>';
            foreach ($warnings as $warning) {
                echo '<li class="warning">' . htmlspecialchars($warning) . '</li>';
            }
            echo '</ul>';
        }
    }
    echo '</div>';
    
    // Quick Fix Commands
    echo '<div class="section">';
    echo '<h2>🔧 Quick Fix Commands</h2>';
    echo '<pre>';
    echo "php artisan key:generate\n";
    echo "php artisan config:clear\n";
    echo "php artisan cache:clear\n";
    echo "php artisan route:clear\n";
    echo "php artisan view:clear\n";
    echo "chmod -R 775 storage bootstrap/cache\n";
    echo "php artisan storage:link\n";
    echo "php artisan config:cache\n";
    echo "php artisan route:cache\n";
    echo '</pre>';
    echo '</div>';
    
    echo '<p class="warning" style="margin-top: 30px; padding: 10px; border: 1px solid #ff0;">';
    echo '⚠️ <strong>SECURITY WARNING:</strong> Delete this file (diagnose.php) immediately after diagnosis!';
    echo '</p>';
    ?>
</body>
</html>
