<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;

class WithdrawController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function store(Request $request)
    {
        try {
            $user = Auth::user();

            // Build validation rules dynamically based on withdrawal method
            $rules = [
                'amount' => 'required|numeric|min:20',
                'method' => 'required|string|in:pix,bank_transfer',
            ];
            
            // Add method-specific validation rules
            if ($request->method === 'pix') {
                $rules['pix_key'] = 'required|string|min:1';
                $rules['pix_type'] = 'required|string|in:document,email,phone,random';
            } elseif ($request->method === 'bank_transfer') {
                $rules['bank_name'] = 'required|string|min:2';
                $rules['account_number'] = 'required|string|min:5';
                $rules['account_holder'] = 'required|string|min:3';
            }
            
            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Check balance
            if ($user->balance < $request->amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance!',
                ], 400);
            }

            // Check if orders table exists
            if (!Schema::hasTable('orders')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Withdrawal service is not yet available. Please try again later.',
                ], 503);
            }

            // Store withdrawal details for reference (could use a separate withdrawals table)
            // For now, we'll use the orders table
            // Withdrawals immediately deduct from balance (status=1 for loss orders)
            $fillable = (new Order())->getFillable();
            $orderData = [
                'user_id' => $user->id,
                'amount' => $request->amount,
                'status' => 1, // Completed - balance deducted immediately (can be refunded if rejected later)
            ];

            // Only add fields that exist in the model's fillable array
            if (in_array('transaction_id', $fillable)) {
                $orderData['transaction_id'] = uniqid('withdraw_');
            }
            if (in_array('game', $fillable)) {
                $orderData['game'] = 'withdrawal';
            }
            if (in_array('game_uuid', $fillable)) {
                $orderData['game_uuid'] = uniqid('withdraw_');
            }
            if (in_array('type', $fillable)) {
                $orderData['type'] = 'loss'; // Withdrawal is a loss from balance
            }
            if (in_array('type_money', $fillable)) {
                $orderData['type_money'] = 'balance_withdrawal';
            }
            if (in_array('providers', $fillable)) {
                $orderData['providers'] = $request->method === 'pix' ? 'pix_withdrawal' : 'bank_withdrawal';
            }
            if (in_array('refunded', $fillable)) {
                $orderData['refunded'] = 0;
            }
            if (in_array('hash', $fillable)) {
                // Store withdrawal details in hash field (or use separate field if available)
                $withdrawalDetails = [];
                if ($request->method === 'pix') {
                    $withdrawalDetails = [
                        'pix_key' => $request->pix_key ?? '',
                        'pix_type' => $request->pix_type ?? 'document',
                    ];
                } elseif ($request->method === 'bank_transfer') {
                    $withdrawalDetails = [
                        'bank_name' => $request->bank_name ?? '',
                        'account_number' => $request->account_number ?? '',
                        'account_holder' => $request->account_holder ?? '',
                    ];
                }
                $orderData['hash'] = md5(json_encode($withdrawalDetails) . uniqid());
            }

            $order = Order::create($orderData);
            
            // Refresh user to get updated balance (balance should be reduced)
            $user = $user->fresh();

            // In a real application, you would integrate with payment processor here
            // For now, we'll create the withdrawal record as pending

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal request submitted successfully. It will be processed within 24-48 hours.',
                'order_id' => $order->id ?? null,
                'balance' => $user->balance,
                'formatted_balance' => $user->formatted_balance,
            ]);
        } catch (\Exception $e) {
            \Log::error('Withdraw error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to process withdrawal request. Please try again.',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }
}
