<?php

namespace App\Http\Controllers;

use App\Models\Deposit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;

class TopupController extends Controller
{
    public function store(Request $request)
    {
        try {
            // Build validation rules dynamically based on payment method
            $rules = [
                'amount' => 'required|numeric|min:10',
                'method' => 'required|string|in:pix,card,bank_transfer',
            ];
            
            // Add method-specific validation rules
            if ($request->method === 'pix') {
                $rules['pix_key'] = 'required|string|min:1';
                $rules['pix_type'] = 'nullable|string|in:document,email,phone,random';
            } elseif ($request->method === 'card') {
                $rules['card_number'] = 'required|string|min:13';
                $rules['card_holder'] = 'required|string|min:3';
                $rules['card_expiry'] = 'required|string|regex:/^\d{2}\/\d{2}$/';
                $rules['card_cvv'] = 'required|string|min:3|max:4';
            } elseif ($request->method === 'bank_transfer') {
                $rules['bank_name'] = 'required|string|min:2';
                $rules['account_number'] = 'required|string|min:5';
                $rules['account_holder'] = 'required|string|min:3';
                $rules['routing_number'] = 'nullable|string';
            }
            
            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Check if deposits table exists
            if (!Schema::hasTable('deposits')) {
                // If table doesn't exist, return success message but don't create record
                return response()->json([
                    'success' => true,
                    'message' => 'Topup request submitted successfully. Please complete the payment.',
                    'note' => 'Database table not yet configured',
                ]);
            }

            // Get fillable columns from Deposit model
            $fillable = (new Deposit())->getFillable();
            
            $depositData = [
                'user_id' => Auth::id(),
                'amount' => $request->amount,
                'status' => 0, // Pending
            ];

            // Only add fields that exist in the model's fillable array
            if (in_array('type', $fillable)) {
                $depositData['type'] = $request->method;
            }
            if (in_array('currency', $fillable)) {
                $depositData['currency'] = 'BRL';
            }
            if (in_array('symbol', $fillable)) {
                $depositData['symbol'] = 'R$';
            }
            
            // Store payment details in proof field if available (for card/bank transfer)
            $paymentDetails = [];
            if ($request->method === 'card') {
                $paymentDetails = [
                    'card_number_last4' => substr(str_replace(' ', '', $request->card_number ?? ''), -4),
                    'card_holder' => $request->card_holder ?? '',
                    'card_expiry' => $request->card_expiry ?? '',
                ];
            } elseif ($request->method === 'bank_transfer') {
                $paymentDetails = [
                    'bank_name' => $request->bank_name ?? '',
                    'account_number' => $request->account_number ?? '',
                    'account_holder' => $request->account_holder ?? '',
                    'routing_number' => $request->routing_number ?? '',
                ];
            } elseif ($request->method === 'pix') {
                $paymentDetails = [
                    'pix_key' => $request->pix_key ?? '',
                    'pix_type' => $request->pix_type ?? 'document',
                ];
            }
            
            if (in_array('proof', $fillable) && !empty($paymentDetails)) {
                $depositData['proof'] = json_encode($paymentDetails);
            }

            // Simulate payment processing
            // For demo purposes, we'll auto-approve card payments
            // In production, this would be handled by payment gateway webhook
            $depositData['status'] = 0; // Default: Pending
            
            // For credit card, simulate successful payment (auto-approve for demo)
            if ($request->method === 'card') {
                // In production, validate card with payment gateway
                // For demo, we'll mark as approved after creating the record
                $depositData['status'] = 1; // Approved (simulated)
            }
            // For PIX and bank transfer, leave as pending (requires manual confirmation)
            
            $deposit = Deposit::create($depositData);
            
            // Build success message based on payment method
            $message = 'Topup request created successfully.';
            if ($request->method === 'card') {
                $message = 'Payment processed successfully! Your account has been credited.';
            } elseif ($request->method === 'bank_transfer') {
                $message = 'Transfer request submitted successfully! Please complete the bank transfer. Your account will be credited once the transfer is confirmed.';
            } elseif ($request->method === 'pix') {
                $message = 'PIX payment initiated! Please complete the payment using the provided QR code or PIX key.';
            }

            // Refresh user to get updated balance
            $user = Auth::user()->fresh();
            
            return response()->json([
                'success' => true,
                'message' => $message,
                'deposit_id' => $deposit->id ?? null,
                'status' => $depositData['status'],
                'balance' => $user->balance,
                'formatted_balance' => $user->formatted_balance,
            ]);
        } catch (\Exception $e) {
            \Log::error('Topup error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to process topup request. Please try again.',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }
}
