<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\FlewAwayMultiplier;
use App\Models\GameRound;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;

class GameRoundController extends Controller
{
    /**
     * Store a game round when it ends
     */
    public function storeRound(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'round_id' => 'required|string',
                'multiplier' => 'required|numeric|min:1',
                'bet_amount' => 'nullable|numeric|min:0',
                'win_amount' => 'nullable|numeric|min:0',
                'result' => 'required|in:win,loss,no_bet',
                'started_at' => 'required|date',
                'ended_at' => 'required|date',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Check if tables exist
            if (!Schema::hasTable('game_rounds') || !Schema::hasTable('flew_away_multipliers')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Database tables not yet configured',
                ], 503);
            }

            $user = Auth::user();

            // Store the game round
            $gameRound = GameRound::updateOrCreate(
                ['round_id' => $request->round_id],
                [
                    'user_id' => $request->result !== 'no_bet' ? $user->id : null,
                    'multiplier' => $request->multiplier,
                    'bet_amount' => $request->bet_amount ?? 0,
                    'win_amount' => $request->win_amount ?? 0,
                    'result' => $request->result,
                    'started_at' => $request->started_at,
                    'ended_at' => $request->ended_at,
                ]
            );

            // Store the flew away multiplier (for all rounds, even if user didn't bet)
            FlewAwayMultiplier::updateOrCreate(
                ['round_id' => $request->round_id],
                [
                    'multiplier' => $request->multiplier,
                    'flew_away_at' => $request->ended_at,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Game round stored successfully',
                'round_id' => $gameRound->round_id,
            ]);
        } catch (\Exception $e) {
            \Log::error('Game round storage error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to store game round',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Get recent flew away multipliers for display
     */
    public function getFlewAwayMultipliers(Request $request)
    {
        try {
            $limit = $request->get('limit', 50);

            if (!Schema::hasTable('flew_away_multipliers')) {
                return response()->json([
                    'success' => true,
                    'multipliers' => [],
                ]);
            }

            $multipliers = FlewAwayMultiplier::orderBy('flew_away_at', 'desc')
                ->limit($limit)
                ->get()
                ->map(function ($item) {
                    return [
                        'multiplier' => floatval($item->multiplier),
                        'flew_away_at' => $item->flew_away_at->toDateTimeString(),
                    ];
                })
                ->toArray();

            return response()->json([
                'success' => true,
                'multipliers' => $multipliers,
            ]);
        } catch (\Exception $e) {
            \Log::error('Get flew away multipliers error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load multipliers',
                'multipliers' => [],
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }
}
