<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Carbon;

class UserManagementController extends Controller
{
    public function index()
    {
        $users = User::orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('admin.users', compact('users'));
    }

    public function show(User $user)
    {
        $deposits = DB::table('deposits')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($deposit) {
                $deposit->created_at = Carbon::parse($deposit->created_at);
                $deposit->updated_at = Carbon::parse($deposit->updated_at);
                return $deposit;
            });
        
        $orders = DB::table('orders')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($order) {
                $order->created_at = Carbon::parse($order->created_at);
                $order->updated_at = Carbon::parse($order->updated_at);
                return $order;
            });
        
        return view('admin.user-detail', compact('user', 'deposits', 'orders'));
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'role_id' => 'required|integer|in:0,1,2',
            'banned' => 'required|boolean',
            'status' => 'required|integer|in:0,1',
        ]);

        if ($request->filled('password')) {
            $validated['password'] = Hash::make($request->password);
        }

        $user->update($validated);

        return redirect()->back()
            ->with('success', 'User updated successfully');
    }

    public function ban(User $user)
    {
        $user->update(['banned' => !$user->banned]);
        
        return redirect()->back()
            ->with('success', $user->banned ? 'User banned' : 'User unbanned');
    }

    public function updateBalance(Request $request, User $user)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric',
            'type' => 'required|in:add,subtract',
        ]);

        $amount = abs($validated['amount']);
        
        if ($validated['type'] === 'add') {
            // Add deposit record
            DB::table('deposits')->insert([
                'user_id' => $user->id,
                'amount' => $amount,
                'status' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        } else {
            // Create withdrawal order
            DB::table('orders')->insert([
                'user_id' => $user->id,
                'amount' => $amount,
                'type' => 'loss',
                'type_money' => 'balance_withdrawal',
                'status' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        return redirect()->back()
            ->with('success', 'Balance updated successfully');
    }

    public function destroy(User $user)
    {
        $user->delete();
        
        return redirect()->route('admin.users')
            ->with('success', 'User deleted successfully');
    }
}
