<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class StatisticsController extends Controller
{
    public function index()
    {
        // Game statistics
        $totalRounds = DB::table('game_rounds')->count();
        $avgMultiplier = DB::table('game_rounds')->avg('multiplier') ?? 0;
        $highestMultiplier = DB::table('game_rounds')->max('multiplier') ?? 0;
        $lowestMultiplier = DB::table('game_rounds')->min('multiplier') ?? 0;

        // Player statistics
        $totalBets = DB::table('orders')
            ->where('type_money', 'balance_withdrawal')
            ->count();
        $totalBetAmount = DB::table('orders')
            ->where('type_money', 'balance_withdrawal')
            ->sum('amount') ?? 0;
        $totalWins = DB::table('orders')
            ->where('type', 'win')
            ->where('type_money', 'balance_withdrawal')
            ->count();
        $totalLosses = DB::table('orders')
            ->where('type', 'loss')
            ->where('type_money', 'balance_withdrawal')
            ->count();

        // Revenue statistics
        $totalRevenue = DB::table('orders')
            ->where('type', 'loss')
            ->where('type_money', 'balance_withdrawal')
            ->where('status', 1)
            ->sum('amount') - DB::table('orders')
            ->where('type', 'win')
            ->where('type_money', 'balance_withdrawal')
            ->where('status', 1)
            ->sum('amount');

        // Flew away multipliers
        $flewAwayMultipliers = DB::table('flew_away_multipliers')
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get()
            ->map(function ($mult) {
                $mult->created_at = Carbon::parse($mult->created_at);
                return $mult;
            });

        // Recent game rounds
        $recentRounds = DB::table('game_rounds')
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get()
            ->map(function ($round) {
                $round->created_at = Carbon::parse($round->created_at);
                return $round;
            });

        // Top winners
        $topWinners = DB::table('orders')
            ->select('orders.user_id', 'users.name', DB::raw('SUM(orders.amount) as total_wins'))
            ->where('orders.type', 'win')
            ->where('orders.type_money', 'balance_withdrawal')
            ->where('orders.status', 1)
            ->leftJoin('users', 'orders.user_id', '=', 'users.id')
            ->groupBy('orders.user_id', 'users.name')
            ->orderBy('total_wins', 'desc')
            ->limit(10)
            ->get();

        // Active users (last 24 hours)
        $activeUsers = DB::table('users')
            ->where('last_activity', '>=', now()->subHours(24))
            ->count();

        // New users (last 7 days)
        $newUsers = DB::table('users')
            ->where('created_at', '>=', now()->subDays(7))
            ->count();

        return view('admin.statistics', compact(
            'totalRounds',
            'avgMultiplier',
            'highestMultiplier',
            'lowestMultiplier',
            'totalBets',
            'totalBetAmount',
            'totalWins',
            'totalLosses',
            'totalRevenue',
            'flewAwayMultipliers',
            'recentRounds',
            'topWinners',
            'activeUsers',
            'newUsers'
        ));
    }

    public function gameHistory(Request $request)
    {
        $query = DB::table('game_rounds')->orderBy('created_at', 'desc');

        if ($request->filled('from')) {
            $query->where('created_at', '>=', $request->from);
        }

        if ($request->filled('to')) {
            $query->where('created_at', '<=', $request->to);
        }

        $rounds = $query->paginate(100);

        return view('admin.game-history', compact('rounds'));
    }
}
