<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class FinanceController extends Controller
{
    public function index()
    {
        $deposits = DB::table('deposits')
            ->join('users', 'deposits.user_id', '=', 'users.id')
            ->orderBy('deposits.created_at', 'desc')
            ->paginate(50, [
                'deposits.*',
                'users.name as user_name',
                'users.email as user_email'
            ]);

        $withdrawals = DB::table('orders')
            ->join('users', 'orders.user_id', '=', 'users.id')
            ->where('orders.type_money', 'balance_withdrawal')
            ->orderBy('orders.created_at', 'desc')
            ->paginate(50, [
                'orders.*',
                'users.name as user_name',
                'users.email as user_email'
            ]);

        // Convert dates to Carbon objects
        $deposits->getCollection()->transform(function ($deposit) {
            $deposit->created_at = Carbon::parse($deposit->created_at);
            $deposit->updated_at = Carbon::parse($deposit->updated_at);
            return $deposit;
        });

        $withdrawals->getCollection()->transform(function ($withdrawal) {
            $withdrawal->created_at = Carbon::parse($withdrawal->created_at);
            $withdrawal->updated_at = Carbon::parse($withdrawal->updated_at);
            return $withdrawal;
        });

        $stats = [
            'total_deposits' => DB::table('deposits')
                ->where('status', 1)
                ->sum('amount'),
            'total_withdrawals' => DB::table('orders')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 1)
                ->sum('amount'),
            'pending_deposits' => DB::table('deposits')
                ->where('status', 0)
                ->sum('amount'),
            'pending_withdrawals' => DB::table('orders')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 0)
                ->sum('amount'),
            'net_profit' => DB::table('orders')
                ->where('type', 'loss')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 1)
                ->sum('amount') - DB::table('orders')
                ->where('type', 'win')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 1)
                ->sum('amount'),
        ];

        return view('admin.finances', compact('deposits', 'withdrawals', 'stats'));
    }

    public function approveDeposit($id)
    {
        $deposit = DB::table('deposits')->where('id', $id)->first();
        
        if (!$deposit) {
            return redirect()->back()->with('error', 'Deposit not found');
        }

        DB::table('deposits')
            ->where('id', $id)
            ->update(['status' => 1]);

        return redirect()->back()
            ->with('success', 'Deposit approved successfully');
    }

    public function rejectDeposit($id)
    {
        $deposit = DB::table('deposits')->where('id', $id)->first();
        
        if (!$deposit) {
            return redirect()->back()->with('error', 'Deposit not found');
        }

        DB::table('deposits')
            ->where('id', $id)
            ->update(['status' => 2]);

        return redirect()->back()
            ->with('success', 'Deposit rejected');
    }

    public function approveWithdrawal($id)
    {
        $withdrawal = DB::table('orders')
            ->where('id', $id)
            ->where('type_money', 'balance_withdrawal')
            ->first();
        
        if (!$withdrawal) {
            return redirect()->back()->with('error', 'Withdrawal not found');
        }

        DB::table('orders')
            ->where('id', $id)
            ->update(['status' => 1]);

        return redirect()->back()
            ->with('success', 'Withdrawal approved successfully');
    }

    public function rejectWithdrawal($id)
    {
        $withdrawal = DB::table('orders')
            ->where('id', $id)
            ->where('type_money', 'balance_withdrawal')
            ->first();
        
        if (!$withdrawal) {
            return redirect()->back()->with('error', 'Withdrawal not found');
        }

        // Refund amount back to user
        DB::table('deposits')->insert([
            'user_id' => $withdrawal->user_id,
            'amount' => $withdrawal->amount,
            'status' => 1,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        DB::table('orders')
            ->where('id', $id)
            ->update(['status' => 2]);

        return redirect()->back()
            ->with('success', 'Withdrawal rejected and amount refunded');
    }
}
